# -*- coding: utf-8 -*-
import os
import re
import time
import json
import threading
import requests
import urllib3
import tkinter as tk
from tkinter import filedialog, messagebox, ttk, scrolledtext, font
from datetime import datetime

# =============================================================================
# ⭐ 基础配置 (用户需自行购买并填写 Key)
# =============================================================================

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# 请在界面中输入 Key，或在此处修改默认值（仅供测试）
DEFAULT_API_URL = "https://................/v1/chat/completions" # 示例，请按需修改
DEFAULT_API_KEY = "" 
DEFAULT_MODEL_NAME = "gpt-4o" 
DEFAULT_MAX_TOKENS = 800

# =============================================================================
# 🌍 国际化资源字典 (I18N) - 12 Language Support
# =============================================================================

UI_STRINGS = {
    'zh': {
        'app_title': "BCAITS 禅宗公案·具身认知与三镜系统 v15.0 (Range Control)",
        'group_api': "API 与 模型配置",
        'lbl_url': "API 地址:",
        'lbl_key': "API Key:",
        'lbl_model': "模型名称:",
        'group_file': "输入源 & 分段控制",
        'btn_select_file': "📂 选择文件",
        'lbl_threshold': "分段阈值 (Tokens):",
        # === 新增 UI 文本 ===
        'btn_gen_split': "📄 仅生成分段预览",
        'chk_range': "启用选段处理",
        'lbl_range_from': "从第",
        'lbl_range_to': "段 到第",
        'lbl_range_end': "段",
        # ===================
        'group_lang': "任务矩阵 (自动生成：split + 独立文件 + 对照版)", 
        'col_lang': "语言",
        'col_trans': "基础翻译",
        'col_exe': "具身三镜解析", 
        'group_log': "系统日志 (进度)",
        'group_preview': "实时预览 (内容)",
        'btn_start': "🚀 启动解析",
        'btn_stop': "🛑 停止并保存",  
        'msg_err_nofile': "错误：请先选择输入文件",
        'msg_err_notask': "错误：请至少勾选一项任务",
        'msg_err_range': "错误：起止段落设置不正确 (必须是数字且 起始<=结束)",
        'msg_stop': "⚠️ 正在停止... (将保存当前进度)", 
        'msg_read_start': "读取文本中...",
        'msg_seg_done': "✅ 智能分段完成：共 {} 个片段。",
        'msg_split_only_done': "✅ 分段预览文件已生成！\n请打开查看段号，然后决定处理范围。\n文件位置: {}",
        'msg_split_created': "已生成原始分段文件: {}",
        'msg_file_created': "创建文件: {}",
        'msg_processing': "处理片段 [第 {} 段] ({}/{}): {}",
        'msg_generating': "  -> [{}] 生成 {} ...",
        'msg_done_title': "完成",
        'msg_done_body': "处理完成！\n已生成 [split] 原文、独立文件及 [对照版] 文件。",
        'msg_stop_title': "已停止",
        'msg_stop_body': "任务已手动停止。\n\n✅ 截至目前的翻译成果已自动保存。\n您可以查看输出文件夹。",
        'msg_skip_timeout': "❌ [超时跳过] 第 {} 段处理失败 (2次尝试均超时/错误)，已标记并跳过。",
        'err_fatal': "❌ 错误: {}",
        'lang_zh': "中文", 'lang_en': "英文", 'lang_ja': "日文",
        'lang_ko': "韩文", 'lang_bo': "藏文", 'lang_pali': "巴利文",
        'lang_es': "西班牙语", 'lang_ru': "俄语", 'lang_fr': "法语", 'lang_de': "德语",
        'lang_hi': "印地语", 'lang_no': "挪威语",
    },
    'en': {
        'app_title': "BCAITS Zen Embodied & 3-Mirror System v15.0 (Range Control)",
        'group_api': "API & Model Settings",
        'lbl_url': "API Endpoint:",
        'lbl_key': "API Key:",
        'lbl_model': "Model Name:",
        'group_file': "Input Source & Segmentation",
        'btn_select_file': "📂 Browse...",
        'lbl_threshold': "Threshold:",
        # === New UI Text ===
        'btn_gen_split': "📄 Generate Split Only",
        'chk_range': "Process Specific Range",
        'lbl_range_from': "From Seg",
        'lbl_range_to': "To Seg",
        'lbl_range_end': "",
        # ===================
        'group_lang': "Task Matrix (Auto-generates Contrast File)",
        'col_lang': "Language",
        'col_trans': "Translation",
        'col_exe': "Embodied Analysis",
        'group_log': "Log",
        'group_preview': "Preview",
        'btn_start': "🚀 EXECUTE",
        'btn_stop': "🛑 STOP & SAVE",
        'msg_err_nofile': "Error: Select file.",
        'msg_err_notask': "Error: Select at least one task.",
        'msg_err_range': "Error: Invalid Range (Start <= End).",
        'msg_stop': "⚠️ Stopping... (Progress saved)",
        'msg_read_start': "Reading text...",
        'msg_seg_done': "✅ Segments: {}.",
        'msg_split_only_done': "✅ Split file generated!\nCheck segment numbers to define range.\nFile: {}",
        'msg_split_created': "Created split file: {}",
        'msg_file_created': "Created: {}",
        'msg_processing': "Processing [Seg {}] ({}/{}): {}",
        'msg_generating': "  -> [{}] Generating {} ...",
        'msg_done_title': "Finished",
        'msg_done_body': "Done! Split file, Separate files and Contrast files generated.",
        'msg_stop_title': "Stopped",
        'msg_stop_body': "Task stopped manually.\n\n✅ Progress up to now has been saved.",
        'msg_skip_timeout': "❌ [TIMEOUT SKIP] Segment {} failed after 2 retries.",
        'err_fatal': "❌ Error: {}",
        'lang_zh': "Chinese", 'lang_en': "English", 'lang_ja': "Japanese",
        'lang_ko': "Korean", 'lang_bo': "Tibetan", 'lang_pali': "Pali",
        'lang_es': "Spanish", 'lang_ru': "Russian", 'lang_fr': "French", 'lang_de': "German",
        'lang_hi': "Hindi", 'lang_no': "Norwegian",
    }
}

# =============================================================================
# 🧠 AI 核心配置：具身认知 + 口语指引 + 三镜 (Core Intelligence)
# =============================================================================

class LangConfig:
    @staticmethod
    def get_trans_prompt(lang_code):
        base = "You are a professional translator of Buddhist texts."
        rules = "Translate the following Zen Koan accurately and smoothly. No commentary. Keep the original title."
        prompts = {
            'zh': f"{base} 请将公案翻译为通俗流畅的现代白话中文。{rules}",
            'en': f"{base} Translate into elegant English. {rules}",
            'ja': f"{base} 現代日本語に翻訳してください。{rules}",
            'ko': f"{base} 현대 한국어로 번역하십시오. {rules}",
            'bo': f"{base} Translate into Tibetan. {rules}",
            'pali': f"{base} Translate into Romanized Pali. {rules}",
            'es': f"{base} Translate into Spanish. {rules}",
            'ru': f"{base} Translate into Russian. {rules}",
            'fr': f"{base} Translate into French. {rules}",
            'de': f"{base} Translate into German. {rules}",
            'hi': f"{base} Translate into Hindi. {rules}",
            'no': f"{base} Translate into Norwegian. {rules}",
        }
        return prompts.get(lang_code, prompts['en'])

    @staticmethod
    def get_exe_prompt(lang_code):
        # ======================================================
        # 🔥 核心指令：具身口语 + 科技三镜 (完全保留)
        # ======================================================
        
        CORE_INSTRUCTION_ZH = """
        [Role]: 你是一位结合了**认知科学（具身认知/生成主义）**与**传统佛学（中观/如来藏）**的禅宗分析师。
        [Core Task]: 不仅分析义理，更要分析**“语言的物理性干预”**。
        
        [Three Mirrors & Embodiment View]:
        1. **中观镜**：缘起性空，破除自性执着。
        2. **如来藏镜**：本具光明，任运自现。
        3. **科技与具身镜**：结合**具身认知 (Embodied Cognition)**、**预测编码 (Predictive Coding)**、**量子效应**与**Transformer涌现机制**。
        
        [Structure Required] (固定、细节版):
        1. 【直译】（忠实原文，准确无误）
        2. 【背景情景】（时代、人物、历史脉络，细节说明）
        3. 【口语与具身指引】（**关键更新**）：
           - **口语机制**：分析公案中的特定口语（如呵斥、俗语、无义语）如何打破常规语义逻辑？
           - **认知干预**：这些语言如何作为“触发器”阻断大脑的预测编码（Predictive Coding）？
           - **具身效应**：分析其如何引发当下的、身体性的觉知（而非纯粹的思维理解）。
        4. 【三镜映照】（强制三部分，每部分细节明确）：
           - 中观镜：缘起性空如何映照？公案哪里落自性执着？
           - 如来藏镜：本具光明如何显现？公案哪里遮蔽/显露？
           - 科学镜：科技类比（量子纠缠/观察者效应/Transformer Attention机制），明确映射公案机制。
        5. 【直指】（一句锋利棒喝，明确点破核心陷阱或突破）
        6. 【偈颂】（科技禅味总结，短促有力）
        """

        CORE_INSTRUCTION_EN = """
        [Role]: You are a Zen Analyst combining **Cognitive Science (Embodied Cognition)** and **Buddhist Doctrine**.
        [Core Task]: Analyze not just the meaning, but the **"Pragmatic Intervention of Spoken Language"**.
        
        [Structure Required]:
        1. [Literal Translation]: Faithful to the original.
        2. [Context]: Historical background and figures.
        3. [Oral Directive & Embodied Guidance] (**Crucial**):
           - **Linguistic Mechanism**: How do specific colloquialisms/shouts/slang in the Koan disrupt standard semantic logic?
           - **Cognitive Intervention**: How do these words act as triggers to interrupt the brain's "Predictive Coding"?
           - **Embodied Effect**: How does this trigger an immediate, bodily awareness (beyond intellectual understanding)?
        4. [Three-Mirror Analysis]:
           - Madhyamaka Mirror: Emptiness & Dependent Origination. Where is the attachment?
           - Tathagatagarbha Mirror: Buddha Nature & Inherent Luminosity. How is it revealed?
           - Science Mirror: Map to **Quantum Mechanics (Observer Effect)** or **AI (Transformer Emergence)**.
        5. [The Direct Point]: A sharp, direct strike at the core trap.
        6. [Verse]: A short, powerful summary combining Zen and Tech.
        """

        if lang_code == 'zh':
            return f"""{CORE_INSTRUCTION_ZH}
            输出格式（严格遵守）：
            【公案】[标题]
            【直译】...
            【背景情景】...
            【口语与具身指引】... (重点分析语言的认知干预作用)
            【三镜映照】
              - 中观镜: ...
              - 如来藏镜: ...
              - 科学镜: ...
            【直指】...
            【偈颂】...
            """
        elif lang_code == 'en':
            return f"""{CORE_INSTRUCTION_EN}
            Output Format:
            [Koan] [Title]
            [Literal Translation] ...
            [Context] ...
            [Oral Directive & Embodied Guidance] ...
            [Three-Mirror Analysis]
              - Madhyamaka Mirror: ...
              - Tathagatagarbha Mirror: ...
              - Science Mirror: ...
            [The Direct Point] ...
            [Verse] ...
            """
        else:
            lang_names = {'ja': 'Japanese', 'ko': 'Korean', 'bo': 'Tibetan', 'pali': 'Romanized Pali', 'fr': 'French', 'de': 'German', 'ru': 'Russian', 'es': 'Spanish', 'hi': 'Hindi', 'no': 'Norwegian'}
            target_lang = lang_names.get(lang_code, 'English')
            return f"""{CORE_INSTRUCTION_EN}
            IMPORTANT: OUTPUT ALL CONTENT IN {target_lang.upper()}.
            
            Output Format ({target_lang}):
            [Koan] ...
            [Translation] ...
            [Context] ...
            [Oral Directive & Embodied Guidance] ...
            [Three-Mirror Analysis] (Madhyamaka / Tathagatagarbha / Science) ...
            [The Direct Point] ...
            [Verse] ...
            """

    @staticmethod
    def get_file_suffix(lang_code, task_type):
        file_lang_code = 'cn' if lang_code == 'zh' else lang_code
        names = {
            'trans': {'zh': '白话翻译', 'en': 'Translation', 'ja': '現代語訳'},
            'exe': {'zh': '具身三镜解经', 'en': 'Embodied_3Mirror', 'ja': '身体性分析'},
            'combined': {'zh': '三镜对照版', 'en': 'Contrast', 'ja': '対照版'}
        }
        func_name = names.get(task_type, {}).get(lang_code, 'Output')
        return f"_{func_name}.{file_lang_code}.txt"

    @staticmethod
    def validate_pali(text):
        clean_text = re.sub(r'[（(][^)）]*[)）]', '', text.replace('\ufeff', ''))
        if re.search(r'[\u0900-\u097F]', clean_text): return False, "Detected Devanagari"
        return True, None

# =============================================================================
# 📜 智能分段
# =============================================================================

class ZenTextProcessor:
    def __init__(self):
        self.header_pattern = re.compile(r'^\s*([^\s]+(?:禅师|和尚|大士|尊者|如来|佛)|[^\s]+(?:卷第.+)|Case\s+\d+|Chapter\s+\d+|第.+则)\s*$')
        self.sentence_end_pattern = re.compile(r'([。！？；.!?;།༎।]+)')

    def preprocess_text(self, full_text):
        if len(full_text) > 1000 and full_text.count('\n') < 10:
            full_text = self.sentence_end_pattern.sub(r'\1\n', full_text)
        return full_text

    def smart_segmentation(self, full_text, max_chars=3000):
        full_text = self.preprocess_text(full_text)
        lines = full_text.split('\n')
        segments = []
        current_title = "Excerpt / 选段"
        current_buffer = []
        current_count = 0
        is_standard_koan = (sum(1 for line in lines[:100] if self.header_pattern.match(line)) > 2)

        for line in lines:
            line = line.rstrip()
            if not line: continue
            should_split = False; new_title = None
            is_header = False
            
            if is_standard_koan:
                if self.header_pattern.match(line) and len(line) < 40: is_header = True
            else:
                if len(line) < 40 and not self.sentence_end_pattern.search(line[-1:]):
                    if current_count > 200: is_header = True
            
            if is_header: should_split = True; new_title = line.strip()
            if current_count + len(line) > max_chars: should_split = True; new_title = new_title if new_title else current_title + " (Part II)"

            if should_split and current_buffer:
                segments.append({"title": current_title, "content": "\n".join(current_buffer)})
                if new_title:
                    current_title = new_title
                    if is_header: current_buffer = []; current_count = 0
                    else: current_buffer = [line]; current_count = len(line)
                else: current_buffer = [line]; current_count = len(line)
            else:
                if is_header: current_title = line.strip()
                else: current_buffer.append(line); current_count += len(line)
        
        if current_buffer: segments.append({"title": current_title, "content": "\n".join(current_buffer)})
        return segments

# =============================================================================
# 🤖 AI 引擎 (升级：超时重试机制)
# =============================================================================

class AiEngine:
    def __init__(self, api_url, api_key):
        self.api_url = api_url
        self.api_key = api_key
    
    def process(self, title, content, system_prompt, model_name, validator=None):
        user_prompt = f"Target Text: {title}\n\nContent:\n{content}"
        messages = [{"role": "system", "content": system_prompt}, {"role": "user", "content": user_prompt}]
        headers = {"Authorization": f"Bearer {self.api_key}", "Content-Type": "application/json"}
        
        # 3. 容错逻辑：尝试 2 次，每次超时 5 分钟 (300秒)
        MAX_RETRIES = 2
        TIMEOUT_SECONDS = 300 
        
        for attempt in range(MAX_RETRIES):
            try:
                payload = {"model": model_name, "messages": messages, "temperature": 0.7, "max_tokens": 4096}
                # 发送请求，带 300s 超时
                resp = requests.post(self.api_url, headers=headers, json=payload, timeout=TIMEOUT_SECONDS)
                
                if resp.status_code != 200: 
                    print(f"API Error {resp.status_code}: {resp.text}")
                    time.sleep(2) # 稍作休息
                    continue # 触发下一次重试

                res = resp.json()['choices'][0]['message']['content']
                
                if validator:
                    valid, msg = validator(res)
                    if not valid: 
                        print(f"Validation Failed: {msg}")
                        continue
                
                return res # 成功返回
                
            except requests.exceptions.Timeout:
                print(f"Timeout (Attempt {attempt+1}/{MAX_RETRIES})")
            except Exception as e:
                print(f"Exception (Attempt {attempt+1}/{MAX_RETRIES}): {str(e)}")
            
            time.sleep(3) # 重试前冷却

        # 如果循环结束还没返回，说明2次都失败
        return "[FAILED: TIMEOUT OR ERROR]"

# =============================================================================
# 🚀 启动器
# =============================================================================

class LanguageLauncher:
    def __init__(self):
        self.root = tk.Tk(); self.root.title("Language Selection"); 
        w,h=550,550; x,y=(self.root.winfo_screenwidth()-w)//2, (self.root.winfo_screenheight()-h)//2
        self.root.geometry(f"{w}x{h}+{x}+{y}"); self.selected_lang=None
        ttk.Label(self.root, text="Select Interface Language\n请选择界面语言", font=("Arial",14), justify=tk.CENTER).pack(pady=20)
        f=ttk.Frame(self.root); f.pack(pady=10)
        langs=[
            ("中文 (Chinese)",'zh'),("English",'en'),
            ("日本語 (Japanese)",'ja'),("한국어 (Korean)",'ko'),
            ("བོད་ཡིག (Tibetan)",'bo'),("Pāḷi (Roman)",'pali'),
            ("Español (Spanish)",'es'),("Русский (Russian)",'ru'),
            ("Français (French)",'fr'),("Deutsch (German)",'de'),
            ("हिन्दी (Hindi)",'hi'),("Norsk (Norwegian)",'no')
        ]
        for i,(n,c) in enumerate(langs): 
            ttk.Button(f,text=n,command=lambda x=c:self.sel(x),width=22).grid(row=i//2,column=i%2,padx=10,pady=10)
    def sel(self,c): self.selected_lang=c; self.root.destroy()
    def run(self): self.root.mainloop(); return self.selected_lang

# =============================================================================
# 🖥️ GUI 主程序 (升级：分段挑选与独立生成)
# =============================================================================

class ZenUniversalApp:
    def __init__(self, root, ui_lang='zh'):
        self.root = root; self.ui_lang = ui_lang
        self.T = UI_STRINGS.get(ui_lang, UI_STRINGS['en'])
        self.root.title(self.T['app_title'])
        self.root.geometry("1200x950")
        self.processor = ZenTextProcessor()
        self.vars_trans = {}; self.vars_exe = {}
        self.is_running = False; self.stop_event = threading.Event()
        self._setup_ui()
        
    def _setup_ui(self):
        top_container = ttk.Frame(self.root, padding=10); top_container.pack(fill=tk.X)
        api_group = ttk.LabelFrame(top_container, text=self.T['group_api'], padding=10)
        api_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        f1 = ttk.Frame(api_group); f1.pack(fill=tk.X, pady=2)
        ttk.Label(f1, text=self.T['lbl_url'], width=10).pack(side=tk.LEFT)
        self.api_url = tk.StringVar(value=DEFAULT_API_URL); ttk.Entry(f1, textvariable=self.api_url).pack(side=tk.LEFT, fill=tk.X, expand=True, padx=5)
        ttk.Label(f1, text=self.T['lbl_key'], width=8).pack(side=tk.LEFT)
        self.api_key = tk.StringVar(value=DEFAULT_API_KEY); ttk.Entry(f1, textvariable=self.api_key, show="*", width=20).pack(side=tk.LEFT, padx=5)
        f2 = ttk.Frame(api_group); f2.pack(fill=tk.X, pady=5)
        ttk.Label(f2, text=self.T['lbl_model'], width=10).pack(side=tk.LEFT)
        self.model_name = tk.StringVar(value=DEFAULT_MODEL_NAME); ttk.Entry(f2, textvariable=self.model_name).pack(side=tk.LEFT, fill=tk.X, expand=True, padx=5)
        
        action_frame = ttk.Frame(top_container, padding=(10, 0, 0, 0)); action_frame.pack(side=tk.RIGHT, fill=tk.Y)
        style = ttk.Style(); style.configure("Big.TButton", font=("Arial", 12, "bold")); style.configure("Stop.TButton", font=("Arial", 10))
        self.btn_start = ttk.Button(action_frame, text=self.T['btn_start'], command=self.start, style="Big.TButton", width=15)
        self.btn_start.pack(side=tk.TOP, fill=tk.BOTH, expand=True, pady=(5, 5), ipady=10)
        self.btn_stop = ttk.Button(action_frame, text=self.T['btn_stop'], command=self.stop, style="Stop.TButton")
        self.btn_stop.pack(side=tk.BOTTOM, fill=tk.X, pady=(0, 5))

        mid_frame = ttk.Frame(self.root, padding=10); mid_frame.pack(fill=tk.X)
        file_group = ttk.LabelFrame(mid_frame, text=self.T['group_file'], padding=10)
        file_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 5))
        
        # 文件选择
        f_box = ttk.Frame(file_group); f_box.pack(fill=tk.X)
        self.file_path = tk.StringVar(); ttk.Entry(f_box, textvariable=self.file_path).pack(side=tk.LEFT, fill=tk.X, expand=True)
        ttk.Button(f_box, text=self.T['btn_select_file'], command=self._select_file).pack(side=tk.LEFT, padx=5)
        
        # 分段阈值 + 仅生成分段按钮
        t_box = ttk.Frame(file_group); t_box.pack(fill=tk.X, pady=5)
        ttk.Label(t_box, text=self.T['lbl_threshold']).pack(side=tk.LEFT)
        self.ts_limit = tk.IntVar(value=DEFAULT_MAX_TOKENS); ttk.Entry(t_box, textvariable=self.ts_limit, width=8).pack(side=tk.LEFT, padx=5)
        ttk.Button(t_box, text=self.T['btn_gen_split'], command=self.generate_split_only).pack(side=tk.LEFT, padx=20) # [NEW]
        
        # 选段处理区域 [NEW]
        r_box = ttk.Frame(file_group); r_box.pack(fill=tk.X, pady=5)
        self.var_use_range = tk.BooleanVar(value=False)
        self.chk_range = ttk.Checkbutton(r_box, text=self.T['chk_range'], variable=self.var_use_range, command=self._toggle_range_inputs)
        self.chk_range.pack(side=tk.LEFT)
        
        ttk.Label(r_box, text=" |  " + self.T['lbl_range_from']).pack(side=tk.LEFT, padx=(10,0))
        self.var_range_start = tk.IntVar(value=1)
        self.ent_start = ttk.Entry(r_box, textvariable=self.var_range_start, width=5, state='disabled')
        self.ent_start.pack(side=tk.LEFT, padx=5)
        
        ttk.Label(r_box, text=self.T['lbl_range_to']).pack(side=tk.LEFT)
        self.var_range_end = tk.IntVar(value=10)
        self.ent_end = ttk.Entry(r_box, textvariable=self.var_range_end, width=5, state='disabled')
        self.ent_end.pack(side=tk.LEFT, padx=5)
        ttk.Label(r_box, text=self.T['lbl_range_end']).pack(side=tk.LEFT)
        
        lang_group = ttk.LabelFrame(mid_frame, text=self.T['group_lang'], padding=10)
        lang_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(5, 0))
        ttk.Label(lang_group, text=self.T['col_lang'], font=("Arial", 9, "bold")).grid(row=0, column=0, sticky=tk.W)
        ttk.Label(lang_group, text=self.T['col_trans'], font=("Arial", 9, "bold")).grid(row=0, column=1)
        ttk.Label(lang_group, text=self.T['col_exe'], font=("Arial", 9, "bold")).grid(row=0, column=2)
        
        # 完整的 12 种语言列表
        order = ['zh', 'en', 'ja', 'ko', 'bo', 'pali', 'es', 'ru', 'fr', 'de', 'hi', 'no']
        for i, key in enumerate(order):
            row = i + 1
            ttk.Label(lang_group, text=self.T.get(f'lang_{key}', key)).grid(row=row, column=0, sticky=tk.W, padx=5, pady=2)
            val_trans = False; val_exe = (key == self.ui_lang) 
            vt = tk.BooleanVar(value=val_trans); self.vars_trans[key] = vt
            ve = tk.BooleanVar(value=val_exe); self.vars_exe[key] = ve
            ttk.Checkbutton(lang_group, variable=vt).grid(row=row, column=1)
            ttk.Checkbutton(lang_group, variable=ve).grid(row=row, column=2)

        main_content = ttk.PanedWindow(self.root, orient=tk.VERTICAL); main_content.pack(fill=tk.BOTH, expand=True, padx=10, pady=(0, 10))
        log_frame = ttk.LabelFrame(main_content, text=self.T['group_log']); main_content.add(log_frame, weight=1)
        self.log_text = scrolledtext.ScrolledText(log_frame, height=8, font=("Consolas", 9), state='normal'); self.log_text.pack(fill=tk.BOTH, expand=True)
        preview_frame = ttk.LabelFrame(main_content, text=self.T['group_preview']); main_content.add(preview_frame, weight=4)
        self.preview_area = scrolledtext.ScrolledText(preview_frame, font=("微软雅黑", 11)); self.preview_area.pack(fill=tk.BOTH, expand=True)
        self.progress = ttk.Progressbar(self.root, mode='determinate'); self.progress.pack(fill=tk.X, padx=10, pady=(0, 5))

    def _select_file(self):
        f = filedialog.askopenfilename(filetypes=[("Text Files", "*.txt")])
        if f: self.file_path.set(f)
        
    def log(self, msg):
        self.log_text.insert(tk.END, f"[{datetime.now().strftime('%H:%M:%S')}] {msg}\n"); self.log_text.see(tk.END)
        
    def _toggle_range_inputs(self):
        st = 'normal' if self.var_use_range.get() else 'disabled'
        self.ent_start.config(state=st); self.ent_end.config(state=st)

    # === 功能 1: 仅生成分段文件 (带段号) ===
    def generate_split_only(self):
        if not self.file_path.get(): messagebox.showerror("Error", self.T['msg_err_nofile']); return
        try:
            input_file = self.file_path.get()
            base_name = os.path.splitext(input_file)[0]
            with open(input_file, 'r', encoding='utf-8') as f: content = f.read()
            segments = self.processor.smart_segmentation(content, self.ts_limit.get())
            
            split_file_path = f"{base_name}_split.txt"
            with open(split_file_path, 'w', encoding='utf-8') as f:
                for i, seg in enumerate(segments):
                    # 增加段号显示，方便用户对照
                    f.write(f"【第 {i+1} 段】 {seg['title']}\n{seg['content']}\n\n{'='*40}\n\n")
            
            self.log(self.T['msg_split_only_done'].format(os.path.basename(split_file_path)))
            messagebox.showinfo("Done", self.T['msg_split_only_done'].format(os.path.basename(split_file_path)))
        except Exception as e:
            messagebox.showerror("Error", str(e))

    def stop(self):
        if self.is_running: self.stop_event.set(); self.log(self.T['msg_stop'])

    def start(self):
        if not self.file_path.get(): messagebox.showerror("Error", self.T['msg_err_nofile']); return
        
        # 2. 检查选段范围逻辑
        r_start, r_end = None, None
        if self.var_use_range.get():
            try:
                s = self.var_range_start.get()
                e = self.var_range_end.get()
                if s <= 0 or e < s: raise ValueError
                r_start, r_end = s, e
            except:
                messagebox.showerror("Error", self.T['msg_err_range']); return
                
        selected_tasks = []
        selected_langs = set()
        for lang in ['zh', 'en', 'ja', 'ko', 'bo', 'pali', 'es', 'ru', 'fr', 'de', 'hi', 'no']:
            if self.vars_trans[lang].get(): 
                selected_tasks.append({'lang': lang, 'type': 'trans'})
                selected_langs.add(lang)
            if self.vars_exe[lang].get(): 
                selected_tasks.append({'lang': lang, 'type': 'exe'})
                selected_langs.add(lang)
        if not selected_tasks: messagebox.showerror("Error", self.T['msg_err_notask']); return
        self.is_running = True; self.stop_event.clear(); self.btn_start.config(state=tk.DISABLED)
        # 传递 range 参数
        threading.Thread(target=self._run_process, args=(selected_tasks, list(selected_langs), r_start, r_end), daemon=True).start()

    def _run_process(self, tasks, active_langs, r_start, r_end):
        input_file = self.file_path.get()
        base_name = os.path.splitext(input_file)[0]
        stopped_by_user = False # 标记是否为用户主动停止
        try:
            self.log(self.T['msg_read_start'])
            with open(input_file, 'r', encoding='utf-8') as f: content = f.read()
            segments = self.processor.smart_segmentation(content, self.ts_limit.get())
            total_segs = len(segments)
            self.log(self.T['msg_seg_done'].format(total_segs))
            
            # 分段文件（如果之前没生成，这里生成一个标准的）
            split_file_path = f"{base_name}_split.txt"
            if not os.path.exists(split_file_path):
                with open(split_file_path, 'w', encoding='utf-8') as f:
                    for i, seg in enumerate(segments): f.write(f"【第 {i+1} 段】 {seg['title']}\n{seg['content']}\n\n{'='*40}\n\n")
                self.log(self.T['msg_split_created'].format(os.path.basename(split_file_path)))

            handles = {}
            # 使用 'a' 模式追加，支持断点续传（如果用户手动管理文件的话）
            for t in tasks:
                suffix = LangConfig.get_file_suffix(t['lang'], t['type'])
                f = open(base_name + suffix, 'a', encoding='utf-8') 
                handles[(t['lang'], t['type'])] = f
                
            for lang in active_langs:
                suffix = LangConfig.get_file_suffix(lang, 'combined')
                f = open(base_name + suffix, 'a', encoding='utf-8')
                handles[(lang, 'combined')] = f

            ai = AiEngine(self.api_url.get(), self.api_key.get())
            
            # 主处理循环
            for i, seg in enumerate(segments):
                seg_num = i + 1
                
                # === 核心逻辑：选段过滤 ===
                if r_start and r_end:
                    if seg_num < r_start: continue # 没到起始段，跳过
                    if seg_num > r_end: break      # 超过结束段，结束循环
                
                # 检查停止
                if self.stop_event.is_set(): stopped_by_user = True; break
                
                title, text = seg['title'], seg['content']
                self.log(self.T['msg_processing'].format(seg_num, seg_num, total_segs, title))
                segment_results = {lang: {'trans': None, 'exe': None} for lang in active_langs}

                for t in tasks:
                    if self.stop_event.is_set(): stopped_by_user = True; break
                        
                    lang, type_ = t['lang'], t['type']
                    prompt = LangConfig.get_trans_prompt(lang) if type_ == 'trans' else LangConfig.get_exe_prompt(lang)
                    validator = LangConfig.validate_pali if lang == 'pali' else None
                    
                    lang_name = self.T.get(f'lang_{lang}', lang)
                    task_display = self.T['col_trans'] if type_ == 'trans' else self.T['col_exe']
                    self.log(self.T['msg_generating'].format(lang_name, task_display))
                    
                    result = ai.process(title, text, prompt, self.model_name.get(), validator)
                    
                    # 容错处理：如果返回的是失败标记
                    if "[FAILED" in result:
                        self.log(self.T['msg_skip_timeout'].format(seg_num))
                        result = f"⚠️ [FAILED] Segment {seg_num} skipped due to API Timeout/Error."
                    
                    handles[(lang, type_)].write(f"【第 {seg_num} 段 - {title}】\n{result}\n\n{'='*60}\n\n"); handles[(lang, type_)].flush()
                    segment_results[lang][type_] = result
                    
                    self.preview_area.delete(1.0, tk.END)
                    self.preview_area.insert(tk.END, f"--- {lang_name} [{type_}] ---\n{title}\n\n{result}")
                
                if stopped_by_user: break

                for lang in active_langs:
                    combined_f = handles[(lang, 'combined')]
                    combined_text = f"【第 {seg_num} 段 - {title}】\n\n--- Source ---\n{text}\n"
                    res_trans = segment_results[lang].get('trans')
                    res_exe = segment_results[lang].get('exe')
                    if res_trans: combined_text += f"\n--- Translation ---\n{res_trans}\n"
                    if res_exe: combined_text += f"\n--- Embodied & 3-Mirror Analysis ---\n{res_exe}\n"
                    combined_text += f"\n{'='*60}\n\n"
                    combined_f.write(combined_text); combined_f.flush()

                self.progress['value'] = (seg_num / total_segs) * 100
            
            # 循环结束后的状态判断
            if stopped_by_user:
                self.log("🛑 任务已手动停止。当前进度已保存。")
                messagebox.showinfo(self.T.get('msg_stop_title', "已停止"), self.T.get('msg_stop_body', "任务已停止。\n进度已保存。"))
            else:
                self.log("DONE!")
                messagebox.showinfo(self.T['msg_done_title'], self.T['msg_done_body'])
                
        except Exception as e:
            self.log(self.T['err_fatal'].format(str(e))); messagebox.showerror("Error", str(e))
        finally:
            if 'handles' in locals():
                for f in handles.values(): 
                    f.flush() # 强制刷新
                    f.close()
            self.is_running = False; self.btn_start.config(state=tk.NORMAL); self.progress['value'] = 0

if __name__ == "__main__":
    launcher = LanguageLauncher()
    selected_lang = launcher.run()
    if selected_lang:
        root = tk.Tk()
        app = ZenUniversalApp(root, ui_lang=selected_lang)
        root.mainloop()
